<?php
ob_start();
header('Content-Type: application/json');

// For debugging (remove in production)
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

session_start();
if (!isset($_SESSION['user_id'])) {
    echo json_encode(['success' => false, 'message' => 'Session expired. Please log in again.']);
    exit;
}
require_once '../config/db.php';
require_once '../auth/auth.php';
require_once '../includes/functions.php';
require_once '../includes/sms_functions.php';
require_once '../includes/whatsapp_functions.php';
require_once '../includes/email_functions.php';

// Check if user has permission to manage suppliers
if (!hasPermission('manage_suppliers')) {
    echo json_encode(['success' => false, 'message' => 'Permission denied']);
    exit;
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        $payment_status = $_POST['payment_status'] ?? null;
        $payment_method = $_POST['payment_method'] ?? null;
        $mpesa_code = $_POST['mpesa_code'] ?? null;
        $distribution = isset($_POST['payment_distribution']) ? json_decode($_POST['payment_distribution'], true) : [];

        if (!$payment_status || empty($distribution)) {
            throw new Exception('Missing required fields');
        }

        $pdo->beginTransaction();

        // Group distribution by supplier_id to avoid duplicate payment records
        $supplier_payments = [];
        foreach ($distribution as $dist) {
            $receipt_id = intval($dist['item_id']);
            $amount = floatval($dist['amount']);

            if (!$receipt_id || $amount <= 0) continue;

            // Get supplier_id for this receipt
            $stmt = $pdo->prepare("SELECT supplier_id FROM supplier_receipts WHERE id = ?");
            $stmt->execute([$receipt_id]);
            $supplier_id = $stmt->fetchColumn();

            if (!isset($supplier_payments[$supplier_id])) {
                $supplier_payments[$supplier_id] = [
                    'total_amount' => 0,
                    'receipts' => []
                ];
            }

            $supplier_payments[$supplier_id]['total_amount'] += $amount;
            $supplier_payments[$supplier_id]['receipts'][$receipt_id] = $amount;
        }

        // Process payments by supplier
        foreach ($supplier_payments as $supplier_id => $payment_data) {
            $total_payment_amount = $payment_data['total_amount'];
            $receipt_payments = $payment_data['receipts'];

            // Calculate new balance after payment
            $stmt = $pdo->prepare("SELECT SUM(amount) FROM supplier_payments WHERE supplier_id = ?");
            $stmt->execute([$supplier_id]);
            $total_paid = $stmt->fetchColumn() ?: 0;
            $stmt = $pdo->prepare("SELECT SUM(quantity * unit_price) FROM supplier_receipts WHERE supplier_id = ?");
            $stmt->execute([$supplier_id]);
            $total_due = $stmt->fetchColumn() ?: 0;
            $balance_after = $total_due - ($total_paid + $total_payment_amount);

            // Insert single payment record per supplier
            $stmt = $pdo->prepare("
                INSERT INTO supplier_payments
                (supplier_id, amount, payment_method, reference, balance_after, created_at, created_by, notes)
                VALUES (?, ?, ?, ?, ?, NOW(), ?, ?)
            ");
            $reference = $mpesa_code ?: ($_POST['cheque_number'] ?? '');
            $notes = $_POST['payment_notes'] ?? '';
            $stmt->execute([
                $supplier_id,
                $total_payment_amount,
                $payment_method,
                $reference,
                $balance_after,
                $_SESSION['user_id'],
                $notes
            ]);

            // Update each receipt's payment status
            foreach ($receipt_payments as $receipt_id => $amount) {
                // Get the total due for this receipt
                $stmt = $pdo->prepare("SELECT quantity * unit_price FROM supplier_receipts WHERE id = ?");
                $stmt->execute([$receipt_id]);
                $total_due_for_receipt = $stmt->fetchColumn() ?: 0;

                // Calculate total paid towards this specific receipt
                // This includes the current payment amount allocated to this receipt
                $stmt = $pdo->prepare("
                    SELECT COALESCE(
                        (SELECT SUM(quantity * unit_price) FROM supplier_receipts WHERE id = ? AND payment_status = 'paid'), 0
                    ) + ?
                ");
                $stmt->execute([$receipt_id, $amount]);
                $total_paid_for_receipt = $stmt->fetchColumn() ?: 0;

                // Determine receipt status - mark as paid only if this payment fully covers the receipt
                $receipt_status = ($amount >= $total_due_for_receipt) ? 'paid' : 'credit';

                // Update supplier_receipts
                $stmt = $pdo->prepare("
                    UPDATE supplier_receipts 
                    SET payment_status = ?,
                        payment_method = ?,
                        payment_details = ?,
                        updated_at = NOW()
                    WHERE id = ?
                ");

                $payment_details = json_encode([
                    'method' => $payment_method,
                    'mpesa_code' => $mpesa_code,
                    'amount' => $amount
                ]);

                $stmt->execute([
                    $receipt_status,
                    $payment_method,
                    $payment_details,
                    $receipt_id
                ]);

                logSystemAction(
                    'supplier_payment_processed',
                    "Payment processed for receipt ID {$receipt_id}: {$receipt_status}, Amount: {$amount}"
                );
            }
        }

        $pdo->commit();

        // After commit, send SMS and email notifications for each supplier involved
        foreach ($supplier_payments as $supplier_id => $payment_data) {
            $total_payment_amount = $payment_data['total_amount'];

            // Fetch supplier info
            $stmt = $pdo->prepare("SELECT * FROM suppliers WHERE id = ?");
            $stmt->execute([$supplier_id]);
            $supplier = $stmt->fetch();
            if (!$supplier) continue;

            // Calculate new balance after payment
            $stmt = $pdo->prepare("SELECT SUM(amount) FROM supplier_payments WHERE supplier_id = ?");
            $stmt->execute([$supplier_id]);
            $total_paid = $stmt->fetchColumn() ?: 0;
            $stmt = $pdo->prepare("SELECT SUM(quantity * unit_price) FROM supplier_receipts WHERE supplier_id = ?");
            $stmt->execute([$supplier_id]);
            $total_due = $stmt->fetchColumn() ?: 0;
            $balance_after = $total_due - $total_paid;

            // Fetch store settings
            $settings = [];
            $stmt = $pdo->query("SELECT setting_key, setting_value FROM store_settings");
            foreach ($stmt->fetchAll() as $row) {
                $settings[$row['setting_key']] = $row['setting_value'];
            }

            // Compose message
            $supplier_name = $supplier['name'];
            $contact_person = $supplier['contact_person'] ?: $supplier_name;
            $to_phone = $supplier['phone'];
            $to_email = $supplier['email'];
            $currency = isset($settings['currency']) ? $settings['currency'] : 'Ksh';
            $store_name = $settings['store_name'] ?? 'Our Store';
            $store_phone = $settings['store_phone'] ?? '';
            $store_email = $settings['store_email'] ?? '';
            
            $msg_amount = $currency . ' ' . number_format($total_payment_amount, 2);
            $msg_balance = $currency . ' ' . number_format($balance_after, 2);
            $is_full = ($balance_after <= 0.01);
            $method = $payment_method;
            $reference = $mpesa_code ?: ($_POST['cheque_number'] ?? '');
            
            // Create notification message
            $msg = "Dear {$contact_person},\n\n";
            $msg .= "A payment of {$msg_amount} has been made to your account via ";
            if ($method === 'mpesa') {
                $msg .= "M-PESA (Ref: {$reference}). ";
            } elseif ($method === 'cheque') {
                $msg .= "Cheque (No: {$reference}). ";
            } else {
                $msg .= ucfirst($method) . ". ";
            }
            if ($is_full) {
                $msg .= "Your account is now fully paid. ";
            } else {
                $msg .= "Remaining balance: {$msg_balance}. ";
            }
            $msg .= "\n\nThank you for your business!\n";
            $msg .= "{$store_name}";
            if ($store_phone) {
                $msg .= "\nPhone: {$store_phone}";
            }
            if ($store_email) {
                $msg .= "\nEmail: {$store_email}";
            }

            // Send SMS notification
            if (!empty($to_phone) && isset($_POST['send_sms']) && $_POST['send_sms'] === '1') {
                try {
                    sendSMS($to_phone, $msg);
                    logSystemAction('supplier_payment_sms_sent', "SMS notification sent to supplier {$supplier_name} for payment of {$msg_amount}");
                    
                    // Save message to supplier_messages table
                    $stmt = $pdo->prepare("
                        INSERT INTO supplier_messages 
                        (supplier_id, message, amount, type, status, created_at) 
                        VALUES (?, ?, ?, 'sms', 'sent', NOW())
                    ");
                    $stmt->execute([$supplier_id, $msg, $total_payment_amount]);
                    
                } catch (Exception $e) {
                    error_log("Failed to send SMS to supplier {$supplier_name}: " . $e->getMessage());
                    
                    // Save failed message to supplier_messages table
                    $stmt = $pdo->prepare("
                        INSERT INTO supplier_messages 
                        (supplier_id, message, amount, type, status, created_at) 
                        VALUES (?, ?, ?, 'sms', 'failed', NOW())
                    ");
                    $stmt->execute([$supplier_id, $msg, $total_payment_amount]);
                }
            }

            // Send WhatsApp notification
            if (!empty($to_phone) && isset($_POST['send_whatsapp']) && $_POST['send_whatsapp'] === '1') {
                try {
                    sendWhatsApp($to_phone, $msg);
                    logSystemAction('supplier_payment_whatsapp_sent', "WhatsApp notification sent to supplier {$supplier_name} for payment of {$msg_amount}");
                    
                    // Save message to supplier_messages table
                    $stmt = $pdo->prepare("
                        INSERT INTO supplier_messages 
                        (supplier_id, message, amount, type, status, created_at) 
                        VALUES (?, ?, ?, 'whatsapp', 'sent', NOW())
                    ");
                    $stmt->execute([$supplier_id, $msg, $total_payment_amount]);
                    
                } catch (Exception $e) {
                    error_log("Failed to send WhatsApp to supplier {$supplier_name}: " . $e->getMessage());
                    
                    // Save failed message to supplier_messages table
                    $stmt = $pdo->prepare("
                        INSERT INTO supplier_messages 
                        (supplier_id, message, amount, type, status, created_at) 
                        VALUES (?, ?, ?, 'whatsapp', 'failed', NOW())
                    ");
                    $stmt->execute([$supplier_id, $msg, $total_payment_amount]);
                }
            }

            // Send Email notification
            if (!empty($to_email) && isset($_POST['send_email']) && $_POST['send_email'] === '1') {
                try {
                    $email_subject = "Payment Notification - {$store_name}";
                    $email_message = "
                        <html>
                        <body>
                            <h2>Payment Notification</h2>
                            <p>Dear {$contact_person},</p>
                            <p>A payment of <strong>{$msg_amount}</strong> has been made to your account via <strong>" . ucfirst($method) . "</strong>.";
                    if ($reference) {
                        $email_message .= " Reference: <strong>{$reference}</strong>.";
                    }
                    $email_message .= "</p>";
                    if ($is_full) {
                        $email_message .= "<p>Your account is now <strong>fully paid</strong>.</p>";
                    } else {
                        $email_message .= "<p>Remaining balance: <strong>{$msg_balance}</strong>.</p>";
                    }
                    $email_message .= "
                            <p>Thank you for your business!</p>
                            <p>Best regards,<br>
                            <strong>{$store_name}</strong>";
                    if ($store_phone) {
                        $email_message .= "<br>Phone: {$store_phone}";
                    }
                    if ($store_email) {
                        $email_message .= "<br>Email: {$store_email}";
                    }
                    $email_message .= "</p>
                        </body>
                        </html>";

                    sendEmail($to_email, $email_subject, $email_message);
                    logSystemAction('supplier_payment_email_sent', "Email notification sent to supplier {$supplier_name} for payment of {$msg_amount}");
                    
                    // Save message to supplier_messages table
                    $stmt = $pdo->prepare("
                        INSERT INTO supplier_messages 
                        (supplier_id, message, amount, type, status, created_at) 
                        VALUES (?, ?, ?, 'email', 'sent', NOW())
                    ");
                    $stmt->execute([$supplier_id, $email_message, $total_payment_amount]);
                    
                } catch (Exception $e) {
                    error_log("Failed to send email to supplier {$supplier_name}: " . $e->getMessage());
                    
                    // Save failed message to supplier_messages table
                    $stmt = $pdo->prepare("
                        INSERT INTO supplier_messages 
                        (supplier_id, message, amount, type, status, created_at) 
                        VALUES (?, ?, ?, 'email', 'failed', NOW())
                    ");
                    $stmt->execute([$supplier_id, $email_message, $total_payment_amount]);
                }
            }


        }

        // Before final JSON output, clean the output buffer
        ob_end_clean();
        header('Content-Type: application/json');
        echo json_encode(['success' => true]);
        exit;
    } catch (Exception $e) {
        if ($pdo->inTransaction()) {
            $pdo->rollBack();
        }
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
        exit;
    }
} else {
    echo json_encode(['success' => false, 'message' => 'Invalid request method']);
    exit;
}

$output = ob_get_clean();
if (!empty($output)) {
    echo json_encode(['success' => false, 'message' => 'Unexpected output: ' . $output]);
    exit;
}